# scripts/compute_action.py
#!/usr/bin/env python3
"""
Compute the discrete master action from the reproduction kernel and save as a pickle.
Reads 'data/kernel.npy', 'data/D_values.csv', and 'data/pivot_params.json'.
"""
import argparse
import pickle
import json
import numpy as np
from ar_sim.continuum_action import ContinuumAction


def main():
    parser = argparse.ArgumentParser(
        description="Compute discrete action from kernel matrix"
    )
    parser.add_argument(
        "--kernel", help="Path to .npy file with kernel matrix", default="data/kernel.npy"
    )
    parser.add_argument(
        "--pivot-params", help="Path to JSON file with pivot params", default="data/pivot_params.json"
    )
    parser.add_argument(
        "--n-values", help="Path to CSV file with n-values (first column)", default="data/D_values.csv"
    )
    parser.add_argument(
        "--out", help="Output pickle for discrete action", default="data/discrete_action.pkl"
    )
    args = parser.parse_args()

    # Load context levels
    # assuming CSV with header 'n,D' -> n is first column
    n_vals = np.loadtxt(args.n_values, delimiter=',', skiprows=1, usecols=0)

    # Load pivot parameters and D_vals
    with open(args.pivot_params, 'r') as f:
        pivot = json.load(f)
    if 'D_vals' not in pivot:
        # load D column if missing
        D_vals = np.loadtxt(args.n_values, delimiter=',', skiprows=1, usecols=1)
        pivot['D_vals'] = D_vals.tolist()

    # Load kernel matrix
    M = np.load(args.kernel)

    # Instantiate ContinuumAction with provided kernel
    cont = ContinuumAction(n_vals=n_vals, pivot_params=pivot)
    # Override internal kernel with loaded M
    cont.M = M

    # Compute discrete action symbolic expression
    S_expr = cont.discrete_action()

    # Save the symbolic action to pickle
    with open(args.out, 'wb') as f:
        pickle.dump(S_expr, f)

    print(f"Saved discrete action expression to {args.out}")

if __name__ == '__main__':
    main()

